﻿﻿using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Collections.Generic;
using System.Linq;

using System.Web.Mvc;

using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

using HIPS.Web.Components.ServiceModel;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.Test.Common;
using HIPS.Web.UI.Controllers;
using HIPS.Web.UI.ViewModels.PcehrView;

using Moq;

namespace HIPS.Web.Test.PcehrView.Controller
{

    /// <summary>
    /// Unit tests for the PcehrViewController.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
    /// </history>
    [TestClass]
    public class PcehrViewControllerTests : TestBase
    {

        #region Methods

        #region Tests

        #region Patients

        /// <summary>
        /// Test that the controller can provide a list of hospitals.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="25 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_List_Hospitals()
        {
            //Arrange
            int numberOfHospitals = 5;
            string selectedHospitalId = null;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(null, 0, 0);
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(null, null, null, 0);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.Patients(selectedHospitalId);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(ViewResult), "Test for expected result type failed.");
            Assert.IsInstanceOfType(((ViewResult)result).Model, typeof(PatientsViewModel), "Test for expected view model type failed.");
            var vm = (PatientsViewModel)((ViewResult)result).Model;
            Assert.IsNull(vm.HospitalId, "Test for null hospital failed.");
            Assert.IsNull(vm.PatientId, "Test for null patient failed.");
            Assert.IsNull(vm.RepositoryId, "Test for null repository failed.");
            Assert.IsNull(vm.DocumentId, "Test for null document failed.");
            Assert.IsTrue(!vm.IsHospitalSelected(), "Test for (not) viewing patient list failed.");
            Assert.IsTrue(!vm.IsPatientSelected(), "Test for (not) viewing document list failed.");
            Assert.IsTrue(!vm.IsDocumentSelected(), "Test for (not) viewing document content failed.");
            mockSettingsRepository.Verify(m => m.GetSettings(), Times.Once, "Test for verifying retrieval of settings from repository failed.");
            mockHospitalRepository.Verify(m => m.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM), Times.AtLeastOnce, "Test for verifying retrieval of hospitals from repository failed.");
            mockPatientRepository.Verify(m => m.ListPatientsCurrentlyInHospital(It.IsAny<string>(), It.IsAny<string>(), It.IsAny<bool?>(), It.IsAny<UserDetails>()), Times.Never, "Test for verifying (non) retrieval of patients from repository failed.");
            Assert.IsNotNull(vm.Hospitals, "Test for non-null list of hospitals failed.");
            Assert.AreEqual(numberOfHospitals, vm.Hospitals.Count(), "Test for expected number of hospitals failed.");

        }

        /// <summary>
        /// Test that the controller can provide a list of patients at a selected hospital.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="20 December 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_List_Patients_At_Hospital()
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(null, null, null, 0);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.Patients(selectedHospitalId);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(ViewResult), "Test for expected result type failed.");
            Assert.IsInstanceOfType(((ViewResult)result).Model, typeof(PatientsViewModel), "Test for expected view model type failed.");
            var vm = (PatientsViewModel)((ViewResult)result).Model;
            Assert.IsNotNull(vm.HospitalId, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.HospitalId, "Test for expected hospital selection failed.");
            Assert.IsNull(vm.PatientId, "Test for null patient failed.");
            Assert.IsNull(vm.RepositoryId, "Test for null repository failed.");
            Assert.IsNull(vm.DocumentId, "Test for null document failed.");
            Assert.IsTrue(vm.IsHospitalSelected(), "Test for viewing patient list failed.");
            Assert.IsTrue(!vm.IsPatientSelected(), "Test for (not) viewing document list failed.");
            Assert.IsTrue(!vm.IsDocumentSelected(), "Test for (not) viewing document content failed.");
            mockSettingsRepository.Verify(m => m.GetSettings(), Times.Once, "Test for verifying retrieval of settings from repository failed.");
            mockHospitalRepository.Verify(m => m.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM), Times.AtLeastOnce, "Test for verifying retrieval of hospitals from repository failed.");
            mockPatientRepository.Verify(m => m.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, It.IsAny<UserDetails>()), Times.AtLeastOnce, "Test for verifying retrieval of patients from repository failed.");
            Assert.IsNotNull(vm.Hospitals, "Test for non-null list of hospitals failed.");
            Assert.AreEqual(numberOfHospitals, vm.Hospitals.Count(), "Test for expected number of hospitals failed.");
            Assert.IsNotNull(vm.Patients, "Test for non-null list of patients failed.");
            Assert.AreEqual(numberOfPatientsWithPcehr, vm.Patients.Count(), "Test for expected number of patients failed.");
            Assert.IsTrue(vm.Patients.All(p => p.HospitalName == selectedHospital.Name), "Test for expected hospital for all patients failed."); // Note: Could fail if we use a different hospital name in the patient view model.
        }

        #endregion

        #region Patient Summary

        /// <summary>
        /// Test that the controller can provide a summary for a selected patient.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="13 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Provide_Summary_For_Patient()
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var selectedPatient = mockPatientRepository.Object.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, TestAssistant.GetCurrentUserDetails()).Data.PatientInHospitalList[2];
            var selectedPatientId = selectedPatient.Mrn;
            var numberOfDocumentsPerClass = 5;
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(selectedHospital, selectedPatient, TestAssistant.GetDocumentClasses(), numberOfDocumentsPerClass, AccessCodeRequired.AccessGranted, false);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.PatientSummary(selectedHospitalId, selectedPatientId);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(ViewResult), "Test for expected result type failed.");
            Assert.IsInstanceOfType(((ViewResult)result).Model, typeof(PatientSummaryViewModel), "Test for expected view model type failed.");
            var vm = (PatientSummaryViewModel)((ViewResult)result).Model;
            Assert.IsNotNull(vm.HospitalId, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.HospitalId, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.CurrentHospital, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.CurrentHospital.Code, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.PatientId, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.PatientId, "Test for expected patient selection failed.");
            Assert.IsNotNull(vm.CurrentPatient, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.CurrentPatient.MRN, "Test for expected patient selection failed.");
            Assert.IsNull(vm.RepositoryId, "Test for null repository failed.");
            Assert.IsNull(vm.DocumentId, "Test for null document failed.");
            Assert.IsTrue(vm.IsHospitalSelected(), "Test for viewing patient list failed.");
            Assert.IsTrue(vm.IsPatientSelected(), "Test for viewing document list failed.");
            Assert.IsTrue(!vm.IsDocumentSelected(), "Test for (not) viewing document content failed.");
            mockSettingsRepository.Verify(m => m.GetSettings(), Times.Once, "Test for verifying retrieval of settings from repository failed.");
            mockHospitalRepository.Verify(m => m.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM), Times.AtLeastOnce, "Test for verifying retrieval of hospitals from repository failed.");
            mockPatientRepository.Verify(m => m.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, It.IsAny<UserDetails>()), Times.AtLeastOnce, "Test for verifying retrieval of patients from repository failed.");
            mockPcehrViewRepository.Verify(m => m.ListActiveDocuments(It.Is<Mrn>(id => (id.Value == selectedPatientId) && (id.HospitalCodeSystem == TestAssistant.HOSPITAL_CODE_SYSTEM) && (id.HospitalCode == selectedHospitalId)), It.IsAny<UserDetails>()), Times.AtLeastOnce, "Test for verifying retrieval of documents from repository failed.");
            var excludedCategories = (mockSettingsRepository.Object.GetSettings().SingleOrDefault(s => string.Equals(s.Code, HIPS.Web.Model.Common.Setting.SettingCodes.PcehrViewPDDocumentClasses.ToString(), StringComparison.OrdinalIgnoreCase)).Value);
            var numberOfExcludedCategories = excludedCategories.Split(',').Count();
            var expectedNumberOfCategories = TestAssistant.GetDocumentClasses().Count - numberOfExcludedCategories;
            Assert.IsNotNull(vm.DocumentCategories, "Test for non-null list of document categories failed.");
            Assert.AreEqual(expectedNumberOfCategories, vm.DocumentCategories.Count(), "Test for expected number of document categories failed.");
            Assert.IsNotNull(vm.Documents, "Test for non-null list of documents failed.");
            Assert.AreEqual(expectedNumberOfCategories * numberOfDocumentsPerClass, vm.Documents.Count(), "Test for expected number of documents failed.");
        }

        /// <summary>
        /// Test that the controller can support an empty summary a selected patient.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="13 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Provide_Summary_For_Patient_Empty()
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var selectedPatient = mockPatientRepository.Object.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, TestAssistant.GetCurrentUserDetails()).Data.PatientInHospitalList[2];
            var selectedPatientId = selectedPatient.Mrn;
            var numberOfDocumentsPerClass = 0;
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(selectedHospital, selectedPatient, TestAssistant.GetDocumentClasses(), numberOfDocumentsPerClass, AccessCodeRequired.AccessGranted, false);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.PatientSummary(selectedHospitalId, selectedPatientId);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(ViewResult), "Test for expected result type failed.");
            Assert.IsInstanceOfType(((ViewResult)result).Model, typeof(PatientSummaryViewModel), "Test for expected view model type failed.");
            var vm = (PatientSummaryViewModel)((ViewResult)result).Model;
            Assert.IsNotNull(vm.HospitalId, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.HospitalId, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.CurrentHospital, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.CurrentHospital.Code, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.PatientId, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.PatientId, "Test for expected patient selection failed.");
            Assert.IsNotNull(vm.CurrentPatient, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.CurrentPatient.MRN, "Test for expected patient selection failed.");
            Assert.IsNull(vm.RepositoryId, "Test for null repository failed.");
            Assert.IsNull(vm.DocumentId, "Test for null document failed.");
            Assert.IsTrue(vm.IsHospitalSelected(), "Test for viewing patient list failed.");
            Assert.IsTrue(vm.IsPatientSelected(), "Test for viewing document list failed.");
            Assert.IsTrue(!vm.IsDocumentSelected(), "Test for (not) viewing document content failed.");
            mockSettingsRepository.Verify(m => m.GetSettings(), Times.Once, "Test for verifying retrieval of settings from repository failed.");
            mockHospitalRepository.Verify(m => m.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM), Times.AtLeastOnce, "Test for verifying retrieval of hospitals from repository failed.");
            mockPatientRepository.Verify(m => m.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, It.IsAny<UserDetails>()), Times.AtLeastOnce, "Test for verifying retrieval of patients from repository failed.");
            mockPcehrViewRepository.Verify(m => m.ListActiveDocuments(It.Is<Mrn>(id => (id.Value == selectedPatientId) && (id.HospitalCodeSystem == TestAssistant.HOSPITAL_CODE_SYSTEM) && (id.HospitalCode == selectedHospitalId)), It.IsAny<UserDetails>()), Times.AtLeastOnce, "Test for verifying retrieval of documents from repository failed.");
            Assert.IsNotNull(vm.DocumentCategories, "Test for non-null list of document categories failed.");
            Assert.AreEqual(0, vm.DocumentCategories.Count(), "Test for expected number of document categories failed.");
            Assert.IsNotNull(vm.Documents, "Test for non-null list of documents failed.");
            Assert.AreEqual(0, vm.Documents.Count(), "Test for expected number of documents failed.");
            Assert.IsNotNull(vm.Messages, "Test for non-null list of messages failed.");
            Assert.AreEqual(1, vm.Messages.Count(), "Test for expected number of messages failed.");
            Assert.AreEqual("No documents are available for the selected patient.", vm.Messages[0].Description, "Test for expected message failed."); 
        }

        #endregion

        #region Document

        /// <summary>
        /// Test that the controller can provide display a selected document.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="14 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Display_Document()
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var selectedPatient = mockPatientRepository.Object.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, TestAssistant.GetCurrentUserDetails()).Data.PatientInHospitalList[2];
            var selectedPatientId = selectedPatient.Mrn;
            var numberOfDocumentsPerClass = 5;
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(selectedHospital, selectedPatient, TestAssistant.GetDocumentClasses(), numberOfDocumentsPerClass, AccessCodeRequired.AccessGranted, false);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();
            var selectedDocument = mockPcehrViewRepository.Object.ListActiveDocuments(new Mrn(selectedPatientId, selectedHospitalId, TestAssistant.HOSPITAL_CODE_SYSTEM), TestAssistant.GetCurrentUserDetails()).Data.DocumentList[2];
            var selectedRepositoryId = selectedDocument.RepositoryUniqueId;
            var selectedDocumentId = selectedDocument.DocumentUniqueId;

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.Document(selectedHospitalId, selectedPatientId, selectedRepositoryId, selectedDocumentId);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(ViewResult), "Test for expected result type failed.");
            Assert.IsInstanceOfType(((ViewResult)result).Model, typeof(PcehrViewModelBase), "Test for expected view model type failed.");
            var vm = (PcehrViewModelBase)((ViewResult)result).Model;
            Assert.IsNotNull(vm.HospitalId, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.HospitalId, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.CurrentHospital, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.CurrentHospital.Code, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.PatientId, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.PatientId, "Test for expected patient selection failed.");
            Assert.IsNotNull(vm.CurrentPatient, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.CurrentPatient.MRN, "Test for expected patient selection failed.");
            Assert.IsNotNull(vm.RepositoryId, "Test for non-null repository failed.");
            Assert.AreEqual(selectedRepositoryId, vm.RepositoryId, "Test for expected repository selection failed.");
            Assert.IsNotNull(vm.DocumentId, "Test for non-null document failed.");
            Assert.AreEqual(selectedDocumentId, vm.DocumentId, "Test for expected document selection failed.");
            Assert.IsNotNull(vm.CurrentDocument, "Test for non-null document failed.");
            Assert.IsTrue(vm.CurrentDocument.RepositoryId == selectedRepositoryId && vm.CurrentDocument.DocumentId == selectedDocumentId, "Test for expected document selection failed.");
            Assert.IsTrue(vm.IsHospitalSelected(), "Test for viewing patient list failed.");
            Assert.IsTrue(vm.IsPatientSelected(), "Test for viewing document list failed.");
            Assert.IsTrue(vm.IsDocumentSelected(), "Test for viewing document content failed.");
            mockSettingsRepository.Verify(m => m.GetSettings(), Times.Once, "Test for verifying retrieval of settings from repository failed.");
            mockHospitalRepository.Verify(m => m.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM), Times.AtLeastOnce, "Test for verifying retrieval of hospitals from repository failed.");
            mockPatientRepository.Verify(m => m.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, It.IsAny<UserDetails>()), Times.AtLeastOnce, "Test for verifying retrieval of patients from repository failed.");
            mockPcehrViewRepository.Verify(m => m.ListActiveDocuments(It.Is<Mrn>(id => (id.Value == selectedPatientId) && (id.HospitalCodeSystem == TestAssistant.HOSPITAL_CODE_SYSTEM) && (id.HospitalCode == selectedHospitalId)), It.IsAny<UserDetails>()), Times.AtLeastOnce, "Test for verifying retrieval of documents from repository failed.");
            //Not actually called, as is called by child action DocumentContent.
            //mockPcehrViewRepository.Verify(m => m.GetDocument(It.Is<Mrn>(id => (id.Value == selectedPatientId) && (id.HospitalCodeSystem == TestAssistant.HOSPITAL_CODE_SYSTEM) && (id.HospitalCode == selectedHospitalId)), It.IsAny<UserDetails>(), It.Is<DocumentRequest>(d => (d.RepositoryUniqueId == selectedRepositoryId) && (d.DocumentUniqueId == selectedDocumentId))), Times.AtLeastOnce, "Test for verifying retrieval of document from repository failed.");
        }

        #endregion

        #region Gain Access

        /// <summary>
        /// Test that the controller can provide a gain access function.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_Default()
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var selectedPatient = mockPatientRepository.Object.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, TestAssistant.GetCurrentUserDetails()).Data.PatientInHospitalList[2];
            var selectedPatientId = selectedPatient.Mrn;
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(selectedHospital, selectedPatient, null, 0);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            string returnUrl = "some-url";

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.GainAccess(selectedHospitalId, selectedPatientId, returnUrl);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(ViewResult), "Test for expected result type failed.");
            Assert.IsInstanceOfType(((ViewResult)result).Model, typeof(GainAccessViewModel), "Test for expected view model type failed.");
            var vm = (GainAccessViewModel)((ViewResult)result).Model;
            Assert.IsNotNull(vm.HospitalId, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.HospitalId, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.CurrentHospital, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.CurrentHospital.Code, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.PatientId, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.PatientId, "Test for expected patient selection failed.");
            Assert.IsNotNull(vm.CurrentPatient, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.CurrentPatient.MRN, "Test for expected patient selection failed.");
            Assert.IsNull(vm.RepositoryId, "Test for null repository failed.");
            Assert.IsNull(vm.DocumentId, "Test for null document failed.");
            Assert.IsNull(vm.AccessCode, "Test for null access code failed.");
            Assert.AreEqual(AccessType.Unspecified, vm.AccessType, "Test for access type (Unspecified) failed.");
            Assert.AreEqual(returnUrl, vm.ReturnUrl, "Test for expected return URL failed.");
        }

        /// <summary>
        /// Test that the controller can provide gain access (with code - valid).
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_With_Code()
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var selectedPatient = mockPatientRepository.Object.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, TestAssistant.GetCurrentUserDetails()).Data.PatientInHospitalList[2];
            var selectedPatientId = selectedPatient.Mrn;
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(selectedHospital, selectedPatient, null, 0);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            string returnUrl = "some-url";
            AccessType accessType = AccessType.WithCode;
            string accessCode = "RAC";

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.GainAccess(selectedHospitalId, selectedPatientId, returnUrl, accessType, accessCode);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(RedirectResult), "Test for expected result type failed.");
            var redirect = ((RedirectResult)result);
            Assert.AreEqual(returnUrl, redirect.Url, "Test for expected return URL failed.");
        }

        /// <summary>
        /// Test that the controller can provide gain access (with code - invalid code).
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_With_Code_Invalid_Code()
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var selectedPatient = mockPatientRepository.Object.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, TestAssistant.GetCurrentUserDetails()).Data.PatientInHospitalList[2];
            var selectedPatientId = selectedPatient.Mrn;
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(selectedHospital, selectedPatient, null, 0);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            string returnUrl = "some-url";
            AccessType accessType = AccessType.WithCode;
            string accessCode = "XXX";

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.GainAccess(selectedHospitalId, selectedPatientId, returnUrl, accessType, accessCode);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(ViewResult), "Test for expected result type failed.");
            Assert.IsInstanceOfType(((ViewResult)result).Model, typeof(GainAccessViewModel), "Test for expected view model type failed.");
            var vm = (GainAccessViewModel)((ViewResult)result).Model;
            Assert.IsNotNull(vm.HospitalId, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.HospitalId, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.CurrentHospital, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.CurrentHospital.Code, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.PatientId, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.PatientId, "Test for expected patient selection failed.");
            Assert.IsNotNull(vm.CurrentPatient, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.CurrentPatient.MRN, "Test for expected patient selection failed.");
            Assert.IsNull(vm.RepositoryId, "Test for null repository failed.");
            Assert.IsNull(vm.DocumentId, "Test for null document failed.");
            Assert.AreEqual(accessCode, vm.AccessCode, "Test for expected access code failed.");
            Assert.AreEqual(accessType, vm.AccessType, "Test for expected access type failed.");
            Assert.AreEqual(returnUrl, vm.ReturnUrl, "Test for expected return URL failed.");
            Assert.IsTrue(vm.Messages.Count(m => m.Level == Components.Common.MessageLevel.Warning || m.Level == Components.Common.MessageLevel.Error) > 0, "Test for warning or error messages failed.");
        }

        /// <summary>
        /// Test that the controller can provide gain access (with code - no code specified).
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_With_Code_No_Code_Specified()
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var selectedPatient = mockPatientRepository.Object.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, TestAssistant.GetCurrentUserDetails()).Data.PatientInHospitalList[2];
            var selectedPatientId = selectedPatient.Mrn;
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(selectedHospital, selectedPatient, null, 0);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            string returnUrl = "some-url";
            AccessType accessType = AccessType.WithCode;
            string accessCode = "";

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.GainAccess(selectedHospitalId, selectedPatientId, returnUrl, accessType, accessCode);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(ViewResult), "Test for expected result type failed.");
            Assert.IsInstanceOfType(((ViewResult)result).Model, typeof(GainAccessViewModel), "Test for expected view model type failed.");
            var vm = (GainAccessViewModel)((ViewResult)result).Model;
            Assert.IsNotNull(vm.HospitalId, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.HospitalId, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.CurrentHospital, "Test for non-null hospital failed.");
            Assert.AreEqual(selectedHospitalId, vm.CurrentHospital.Code, "Test for expected hospital selection failed.");
            Assert.IsNotNull(vm.PatientId, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.PatientId, "Test for expected patient selection failed.");
            Assert.IsNotNull(vm.CurrentPatient, "Test for non-null patient failed.");
            Assert.AreEqual(selectedPatientId, vm.CurrentPatient.MRN, "Test for expected patient selection failed.");
            Assert.IsNull(vm.RepositoryId, "Test for null repository failed.");
            Assert.IsNull(vm.DocumentId, "Test for null document failed.");
            Assert.AreEqual(accessCode, vm.AccessCode, "Test for expected access code failed.");
            Assert.AreEqual(accessType, vm.AccessType, "Test for expected access type failed.");
            Assert.AreEqual(returnUrl, vm.ReturnUrl, "Test for expected return URL failed.");
            Assert.IsFalse(pcehrViewController.ModelState.IsValid, "Test for (in)valid model state failed.");
        }

        /// <summary>
        /// Test that the controller can provide gain access (emergency).
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_Emergency()
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var selectedPatient = mockPatientRepository.Object.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, TestAssistant.GetCurrentUserDetails()).Data.PatientInHospitalList[2];
            var selectedPatientId = selectedPatient.Mrn;
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(selectedHospital, selectedPatient, null, 0);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            string returnUrl = "some-url";
            AccessType accessType = AccessType.Emergency;
            string accessCode = "";

            //Act
            var pcehrViewController = new PcehrViewController(mockHospitalRepository.Object, mockPatientRepository.Object, mockPcehrViewRepository.Object, mockSettingsRepository.Object);
            var result = pcehrViewController.GainAccess(selectedHospitalId, selectedPatientId, returnUrl, accessType, accessCode);

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.IsInstanceOfType(result, typeof(RedirectResult), "Test for expected result type failed.");
            var redirect = ((RedirectResult)result);
            Assert.AreEqual(returnUrl, redirect.Url, "Test for expected return URL failed.");
        }

        #endregion

        #endregion

        #region Helpers


        #endregion

        #endregion

    }

}
